// External Dependencies
import $ from 'jquery'
// Internal Dependencies
import fields from './fields'
import modules from './modules'

/* A prefix and suffix for video ids which make YouTube ids invalid. Serves
as a way to check, whether an iframe's src attribute requires updating. */
const vimeoVideoIdPrefix = '%%START_OF_VIMEO_VIDEO_ID%%'
const vimeoVideoIdSuffix = '%%END_OF_VIMEO_ID%%'
const localStorageKey = 'cordMediaAcademyDiviHelpVideos'
const restApiUri = 'https://cordmedia.academy/wp-json/cord-media-academy/'
	+ 'divi-help-videos'

let hasFetchedVideos = false
let hasInjectedVideos = false
let rootIframe = null
let fetchedVideos = null

/* MutationObserver to observe iframes for src attributes that we may need
to correct. */
const iframeMutationObserver = new MutationObserver(mutations => {
	mutations.forEach(mutation => {
		modifyIframe(mutation.target)
	})
})

/* MutationObserver to await the builder's root element, new help modals and
video overlays. */
const builderMutationObserver = new MutationObserver(mutations => {
	mutations.forEach(mutation => {
		mutation.addedNodes.forEach(node => {
			if(hasFetchedVideos && !hasInjectedVideos) {
				prepareVideosForInjection()
				hasInjectedVideos = injectVideos()
			}

			if(isFrontendBuilderRoot(node)) {
				rootIframe = $(node).find('iframe')[0]
			} else if(isHelpModal(node) || isVideoOverlay(node)) {
				observeIframesWithinNode(node)
			}
		})
	})
})

function isWindowInIframe() {
    try {
        return window.self !== window.top
    } catch(e) {
        return true
    }
}

function isBackendBuilderIframe(id) {
	return id === 'et-bfb-app-frame'
}

function isThemeBuilderIframe(id) {
	return id === 'et-fb-app-frame'
		&& window.parent.document.body.classList.contains(
			'divi_page_et_theme_builder'
		)
}

function hasVideoStorageExpired() {
	const timestamp = localStorage.getItem(localStorageKey + 'Timestamp')
	if(timestamp === null) {
		return true
	}

	const parsedTimestamp = parseInt(timestamp, 10)
	if(isNaN(parsedTimestamp)) {
		return true
	}

	// Refresh at least every twelve hours
	const cutOffTimestamp = Date.now() - 1000 * 60 * 60 * 12
	return timestamp <= cutOffTimestamp
}

function getVideosFromStorage() {
	if(hasVideoStorageExpired()) {
		return null
	}

	const videos = localStorage.getItem(localStorageKey)
	if(videos === null) {
		return null
	}

	return JSON.parse(videos)
}

async function fetchVideos() {
	if(hasFetchedVideos) {
		return
	}

	// If available, load videos from storage
	fetchedVideos = getVideosFromStorage()
	if(fetchedVideos !== null) {
		hasFetchedVideos = true
		return
	}

	// Fetch videos via REST API
	try {
		const response = await fetch(restApiUri, { redirect: 'follow' })
		if(!response.ok) {
			console.log(
				'Failed to fetch CORD MEDIA Academy Divi help videos.'
			)

			return
		}

		fetchedVideos = await response.json()
		localStorage.setItem(
			localStorageKey,
			JSON.stringify(fetchedVideos)
		)

		localStorage.setItem(localStorageKey + 'Timestamp', Date.now())
		hasFetchedVideos = true
	} catch(error) {
		console.log(
			'Unknown error whilst attempting to fetch CORD MEDIA Academy '
			+ 'Divi help videos.'
		)
	}
}

function prepareVideosForInjection() {
	Object.entries(fetchedVideos).forEach(([module, moduleVideos]) => {
		moduleVideos.forEach((video, index) => {
			if(
				typeof video.originalId === 'undefined'
				&& typeof video.index === 'undefined'
			) {
				return
			}

			if(
				fetchedVideos[module][index].id.startsWith(
					vimeoVideoIdPrefix
				)
			) {
				return
			}

			fetchedVideos[module][index].id = vimeoVideoIdPrefix
				+ fetchedVideos[module][index].id
				+ vimeoVideoIdSuffix
		})
	})
}

function mergeVideos(originalVideos, videos) {
	Object.entries(videos).forEach(([module, moduleVideos]) => {
		if(typeof originalVideos[module] === 'undefined') {
			originalVideos[module] = moduleVideos
			return
		}

		moduleVideos.forEach(video => {
			// If originalId is set and not empty, replace video
			if(
				typeof video.originalId !== 'undefined'
				&& video.originalId.length > 0
			) {
				for(let i = 0; i < originalVideos[module].length; i++) {
					if(originalVideos[module][i].id === video.originalId) {
						originalVideos[module][i] = video
						return
					}
				}
			}

			// If originalId is not set but index is, insert video
			if(typeof video.index !== 'undefined') {
				// Required to know that we need to switch to Vimeo later on
				originalVideos[module].splice(video.index, 0, video)
				return
			}

			// If originalId and index are not set, replace name of video
			for(let i = 0; i < originalVideos[module].length; i++) {
				if(originalVideos[module][i].id === video.id) {
					originalVideos[module][i].name = video.name
					return
				}
			}
		})
	})
}

function injectVideos() {
	if(rootIframe === null) {
		return false
	}

	const iframeWindow = rootIframe.contentWindow || rootIframe
	if(
		typeof iframeWindow === 'undefined'
		|| typeof iframeWindow.ETBuilderBackend === 'undefined'
	) {
		return false
	}

	mergeVideos(
		iframeWindow.ETBuilderBackend.i18n.videos,
		fetchedVideos
	)

	return true
}

function modifyIframe(node) {
	// Enable fullscreen button
	const iframe = $(node)
	iframe.attr('allowfullscreen', '')

	// Modify src if necessary
	if(!node.hasAttribute('src')) {
		return
	}

	const src = iframe.attr('src')
	const prefixPosition = src.indexOf(vimeoVideoIdPrefix)
	if(prefixPosition === -1) {
		return
	}

	// Find vimeo video id in invalid YouTube src
	let vimeoId = src.substr(prefixPosition + vimeoVideoIdPrefix.length)
	vimeoId = vimeoId.substr(0, vimeoId.indexOf(vimeoVideoIdSuffix))

	// Update the iframe's src attribute
	$(node).attr('src', 'https://player.vimeo.com/video/' + vimeoId)
}

function observeIframe(iframe) {
	// Initially, modify the iframe
	modifyIframe(iframe)

	// Start observing the iframe for future changes
	iframeMutationObserver.observe(iframe, { attributeFilter: ['src'] })
}

function isFrontendBuilderRoot(node) {
	return $(node).is('#et_pb_root')
}

function isHelpModal(node) {
	return $(node).hasClass('et-fb-modal--help')
}

function isVideoOverlay(node) {
	return $(node).hasClass('et-fb-video-overlay')
}

function observeIframesWithinNode(node) {
	const iframes = $(node).find('iframe')
	iframes.each((_index, iframe) => {
		observeIframe(iframe)
	})
}

function initBuilder() {
	// Asynchronously fetch videos
	fetchVideos()
}

function initFrontendBuilder() {
	// Observe as soon as possible to detect root iframe
	builderMutationObserver.observe(
		document.getElementById('et-fb-app'),
		{ childList: true }
	)

	// Initialize the rest of the builder
	initBuilder()
}

function initFrontendBuilderInParent() {
	rootIframe = window.frameElement
	builderMutationObserver.observe(
		window.parent.document.getElementById('et-fb-app'),
		{ childList: true }
	)

	// Initialize the rest of the builder
	initBuilder()
}

/* When using the Divi frontend builder, Divi loads the entire page into an
iframe to build the builder around it. To replace videos, we only need to work
inside the builder i.e. outside of the iframe. */
if(!isWindowInIframe()) {
	initFrontendBuilder()
} else {
	/* If this script is only executed inside an iframe, it is most likely not
	the frontend builder but the backend or theme builder instead. */
	if(
		isBackendBuilderIframe(window.frameElement.id)
		|| isThemeBuilderIframe(window.frameElement.id)
	) {
		initFrontendBuilderInParent()
	}
}

// Register modules and fields
$(window).on('et_builder_api_ready', (event, API) => {
	API.registerModules(modules)
	API.registerModalFields(fields)
})